//+------------------------------------------------------------------+
//|                                           GurooEx02Multi2010.mq5 |
//|     Copyright  2010, Marketing Dreams Ltd. All Rights Reserved. |
//|                                     http://raytherandomrobot.com |
//|                                                                  |
//| GuruTrader example 2                                            |
//| Object Oriented MQL5 Version 1.0                                 |
//| Version used in Automated Trading Championship 2010              |
//|                                                                  |
//| A port of our MQL4 robot example 2 to MQL5 for use with          |
//| the public beta test version of MetaTrader 5. It is written      |
//| using MetaQuotes object-oriented framework for MQL5.             |
//|                                                                  |
//| Stays in the market almost 100% of the time. For easier          |
//| comparisons between different platforms this version doesn't     |
//| enter in the direction of the last tick. Instead it checks       |
//| whether the previous bar closed higher or lower than it opened.  |
//| It can be amazingly profitable given that it uses very nearly    |
//| random entries!                                                  |                                                                  |
//|                                                                  |
//| Wealth Warning! This expert is for educational purposes only.    |
//| It should NEVER be used on a live account. Past performance is   |
//| in no way indicative of future results!                          |
//+------------------------------------------------------------------+
#property copyright   "Copyright  2010, Marketing Dreams Ltd."
#property link        "http://raytherandomrobot.com"
#property version     "1.00"
#property description "Ray the Random Robot upgraded with Multiple Micro Memory Modules"

#define MAX_PAIRS 12

//---- include object oriented framework
#include <Trade\Trade.mqh>
#include <Trade\SymbolInfo.mqh>
#include <Trade\PositionInfo.mqh>
#include <Trade\DealInfo.mqh>
#include <Trade\AccountInfo.mqh>
#include <Indicators\Indicators.mqh>

//---- input parameters
input  int     Magic = 4211689;
input  int     Slippage=10;
input  int     ProfitTarget=1000;  
input  int     StopLoss=1000;  
input  double  Lots=0.1;
input  bool    useLog=false;
input  bool    useTicks=false;
input  bool    useTimer=true;
input  bool    useMMMM=true;  
input  bool    useEU=true;
input  bool    useGU=true; 
input  bool    useUJ=true; 
input  bool    useUC=true;
input  bool    useAU=false; 
input  bool    useEG=true; 
input  bool    useEA=false;
input  bool    useES=false; 
input  bool    useEJ=true; 
input  bool    useGJ=true;
input  bool    useGS=true; 
input  bool    useUS=true; 
   
//+------------------------------------------------------------------+
//| Pseudo Random entry example expert class                         |
//+------------------------------------------------------------------+
class CGuruEx02
{
private:
   int               Dig;
   double            Points;
   bool              Initialized;
   bool              Running;
   bool              lastDirn;
   ulong             OrderNumber;

protected:
   string            m_Pair;                    // Currency pair to trade 
   CTrade            m_Trade;                   // Trading object
   CSymbolInfo       m_Symbol;                  // Symbol info object
   CPositionInfo     m_Position;                // Position info object
   void              InitSystem();
   bool              CheckEntry();
   bool              CheckExit();
   string            GetPair();
   bool              isNewBar();
   double            MartingaleSize(int& Target);
   double            OpenPrices[2];
   
public:
                     CGuruEx02();               // Constructor
                    ~CGuruEx02() { Deinit(); }  // Destructor
   bool              Init(string Pair);
   void              Deinit();
   bool              Validated();
   bool              Execute();
};

CGuruEx02 Systems[MAX_PAIRS];

//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CGuruEx02::CGuruEx02()
{
   Initialized = false;
}

string CGuruEx02::GetPair()
{
   return m_Pair;                // Symbol
}
   
//+------------------------------------------------------------------+
//| Return true if a new bar appears for the symbol/period pair      |
//+------------------------------------------------------------------+
bool CGuruEx02::isNewBar()
{
//--- remember the time of opening of the last bar in the static variable
   static datetime last_time=0;
//--- current time 

   datetime lastbar_time=SeriesInfoInteger(m_Pair, Period(), SERIES_LASTBAR_DATE);

//--- if it is the first call of the function
   if(last_time==0)
     {
      //--- set time and exit
      last_time=lastbar_time;
      return(false);
     }

//--- if the time is different
   if(last_time!=lastbar_time)
     {
      //--- memorize time and return true
      last_time=lastbar_time;
      return(true);
     }
//--- if we pass to this line then the bar is not new, return false
   return(false);
}
  
//+------------------------------------------------------------------+
//| Calculates a Martingale style position size                      |
//+------------------------------------------------------------------+
double CGuruEx02::MartingaleSize(int& Target)
{
   int Index, Size;
   double TickSize, TickValue;
   double Profit, CurrLoss, PipLoss, RetVal;
   CDealInfo Deal;
   
   Size = 1;
   
   if (useMMMM) {
      CurrLoss = 0;
   
      HistorySelect(0,TimeCurrent()); // load all history
      if (useLog)
         Print(m_Pair, " - Found ", HistoryDealsTotal(), " deals");
      
      for (Index = HistoryDealsTotal() - 1; Index >= 0; Index--) {
         Deal.Ticket(HistoryDealGetTicket(Index));
         if ((Deal.Symbol() == m_Pair) /* && (Deal.Magic() == Magic) */) {
            TickValue = m_Symbol.TickValue();

            if (useLog)
               Print(m_Pair, " - Found deal ", Deal.Ticket(), ", Magic = ", Deal.Magic(), ", Profit = ", Deal.Profit(), ", Tick Value = ", TickValue);
            if (Deal.Profit()  <= 0) {
               CurrLoss = CurrLoss - Deal.Profit();  // Add up our previous losses in pips
               PipLoss = CurrLoss / (Lots * TickValue);
               Size = 1 + MathRound((PipLoss  - (ProfitTarget / 4)) / ProfitTarget);
            }
            else {
               break;                        // Found a profitable trade, so we're all done 
            }
            
         }
      }
      Print(m_Pair, " - Loss = ", CurrLoss, ", Pips = ", PipLoss, ", Size = ", Size);
   }
      
   RetVal = Lots * Size;
   if (RetVal > 5.0) {
      Target *= RetVal / 5.0;             // Competition limitation
      RetVal = 5.0;                       // Adjust size and target
   }
   
   return (RetVal);
}

//+------------------------------------------------------------------+
//| Performs system initialisation                                   |
//+------------------------------------------------------------------+
bool CGuruEx02::Init(string Pair)
{
   m_Pair = Pair;
   double TickSize;

   if (Pair != "") {   
      m_Symbol.Name(m_Pair);                // Symbol
      m_Trade.SetExpertMagicNumber(Magic);  // Magic number
      lastDirn = false;
      Dig = m_Symbol.Digits();
      Points = m_Symbol.Point();
      TickSize = m_Symbol.TickSize();

      m_Trade.SetDeviationInPoints(Slippage);
      
      Print(m_Pair, " - Period = ", Period(), ", Digits = ", Dig, ", Tick Size = ", DoubleToString(TickSize, Dig), ", Points = ", DoubleToString(Points, Dig));
   
      InitSystem();
   
      Initialized = true;
   }
   
   return(true);
}
         
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
CGuruEx02::Deinit()
{
   Initialized = false;
   
   Print(m_Pair, " - DeInitialized OK");
}

//+------------------------------------------------------------------+
//| Checks if everything initialized successfully                    |
//+------------------------------------------------------------------+
bool CGuruEx02::Validated()
{
   return (Initialized);
}

//+------------------------------------------------------------------+
//| Performs system reinitialisation                                 |
//+------------------------------------------------------------------+
void CGuruEx02::InitSystem()
{
   Running = false;
   
   CopyOpen(m_Pair, Period(), 0, 2, OpenPrices);
   
   Initialized = true;
}
         
//+------------------------------------------------------------------+
//| Performs system logic. Called on every tick                      |
//+------------------------------------------------------------------+
bool CGuruEx02::Execute()   
{
   CopyOpen(m_Pair, Period(), 0, 2, OpenPrices);

   if (Running) {                   // Are we in a trade at the moment?
      if (CheckExit() > 0) {        // Yes - Last trade complete?
         Initialized = false;       // Yes - Indicate we need to reinitialise
         InitSystem();              //  and start all over again!
      }
      isNewBar();                   // Keep it updated, though we don't check return value
   }
   else if (isNewBar()) {
      if (useLog)
         Print(m_Pair, " - New bar at ", TimeToString(TimeCurrent(), TIME_DATE | TIME_SECONDS));
      if (CheckEntry() > 0) {       // Entered a trade?
         Running = true;            // Yes - Indicate that we're in a trade
      }   
   }
   return(true);
}

//+------------------------------------------------------------------+
//| Checks for entry to a trade                                      |
//+------------------------------------------------------------------+
bool CGuruEx02::CheckEntry()
{
   double Size;
   bool   Short;
   int Target = ProfitTarget;
   
   m_Symbol.RefreshRates();
   Size = MartingaleSize(Target);
   Print(m_Pair, " - CheckEntry: This Open = ", OpenPrices[0],
                 ", Last Open = ", OpenPrices[1],
                 ", Size = ", Size,
                 " at ", TimeToString(TimeCurrent(), TIME_DATE | TIME_SECONDS));
   if (OpenPrices[0] > OpenPrices[1])         // Short term uptrend so GO LONG!
      Short = false;
   else if (OpenPrices[0] < OpenPrices[1])    // Short term downtrend so GO SHORT!
      Short = true;
   else
      Short = lastDirn;
      
   if (Short) {
      if (m_Trade.PositionOpen(m_Pair, ORDER_TYPE_SELL, Size, m_Symbol.Bid(),
                               m_Symbol.Bid() + (Points * StopLoss),
                               m_Symbol.Bid() - (Points * Target))) {
         OrderNumber = m_Trade.ResultOrder();
         Print(m_Pair, " - Entered SHORT at ", TimeToString(TimeCurrent(), TIME_DATE | TIME_SECONDS));
         lastDirn = true;
         return(true);
      }   
      else {
         OrderNumber = 0;
      }   
   }
   else {
         if (m_Trade.PositionOpen(m_Pair, ORDER_TYPE_BUY, Size, m_Symbol.Ask(),
                               m_Symbol.Ask() - (Points * StopLoss),
                               m_Symbol.Ask() + (Points * Target))) {
         OrderNumber = m_Trade.ResultOrder();
         Print(m_Pair, " - Entered LONG at ", TimeToString(TimeCurrent(), TIME_DATE | TIME_SECONDS));
         lastDirn = false;
         return(true);
      }
      else {
         OrderNumber = 0;
      }   
   }

   return(false);
}

//+------------------------------------------------------------------+
//| Checks for exit from a trade                                     |
//+------------------------------------------------------------------+
bool CGuruEx02::CheckExit()
{
   double PositionSize;
   
   if (m_Position.Select(m_Pair) != true) {
      Print(m_Pair, " - Position flat at ", TimeToString(TimeCurrent(), TIME_DATE | TIME_SECONDS));
      return(true);
   }
   else {
      m_Position.InfoDouble(POSITION_VOLUME, PositionSize);
      if (PositionSize <= 0) {
         Print(m_Pair, " - Position closed at ", TimeToString(TimeCurrent(), TIME_DATE | TIME_SECONDS));
         return(true);
      }
      else {
         return(false);
      }
   }
}

//+------------------------------------------------------------------+
//| Initialises system for one pair                                  |
//+------------------------------------------------------------------+
bool InitPair(int& i, string Pair)
{
   if (!Systems[i].Init(Pair)) {
      Systems[i].Deinit();
      return false;   
   }
   else {
      i++;
   }
   
   return false;   
}

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   int i = 0;

   if (AccountInfoInteger(ACCOUNT_TRADE_MODE) == ACCOUNT_TRADE_MODE_REAL) {
      MessageBox("Wealth Warning! This expert is for educational purposes only." +
            " It should NEVER be used on a live account." +
            " Past performance is in no way indicative of future results!" +
            " http://community.trading-gurus.com");
      Print("Initialization Failure");
      return(-1);
   }

   if (useEU) {
      InitPair(i, "EURUSD");
   }
   if (useGU) {
      InitPair(i, "GBPUSD");
   }
   if (useUJ) {
      InitPair(i, "USDJPY");
   }
   if (useUC) {
      InitPair(i, "USDCAD");
   }
   if (useAU) { 
      InitPair(i, "AUDUSD");
   }
   if (useEG) { 
      InitPair(i, "EURGBP");
   }
   if (useEA) {
      InitPair(i, "EURAUD");
   }
   if (useES) { 
      InitPair(i, "EURCHF");
   }
   if (useEJ) { 
      InitPair(i, "EURJPY");
   }
   if (useGJ) {
      InitPair(i, "GBPJPY");
   }
   if (useGS) { 
      InitPair(i, "GBPCHF");
   }
   if (useUS) { 
      InitPair(i, "USDCHF");
   }
   for (;i < MAX_PAIRS; i++)
      Systems[i].Init("");
      
   if (useTimer)
      EventSetTimer(1);
         
   Print("Initialized OK");
   
   return(0);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   int i;
   
   for (i = 0; i < MAX_PAIRS; i++) {
      if (Systems[i].GetPair() == "")
         continue;
      Systems[i].Deinit();
   }   
}

void OnTimer()
{
   int i;

   if (!useTimer)
      return;
      
   for (i = 0; i < MAX_PAIRS; i++) {
      if (Systems[i].GetPair() == "")
         continue;
      if (!Systems[i].Validated()) {
         continue;
      }
   
      Systems[i].Execute();
   }
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   int i;

   if (!useTicks)
      return;
      
   for (i = 0; i < MAX_PAIRS; i++) {
      if (Systems[i].GetPair() == "")
         continue;
      if (!Systems[i].Validated()) {
         continue;
      }
   
      Systems[i].Execute();
   }
}
